package com.devrant.android.ui

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import android.widget.*
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.devrant.android.*

class MainActivity : Activity() {
    private lateinit var api: DevRantApi
    private lateinit var repository: RantRepository
    private lateinit var feedViewModel: FeedViewModel
    private lateinit var recyclerView: RecyclerView
    private lateinit var adapter: RantAdapter
    private lateinit var progressBar: ProgressBar
    private lateinit var errorText: TextView
    private lateinit var sortSpinner: Spinner
    private lateinit var loadMoreButton: Button
    private var isLoadingMore = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(createMainLayout())

        api = DevRantApi()
        repository = RantRepository(api)
        feedViewModel = FeedViewModel(repository)

        setupUI()
        loadFeed()
    }

    private fun createMainLayout(): View {
        val container = LinearLayout(this).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = ViewGroup.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
        }

        val topBar = LinearLayout(this).apply {
            orientation = LinearLayout.HORIZONTAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                100
            ).apply { setMargins(10, 10, 10, 10) }
            setBackgroundColor(0xFFEEEEEE.toInt())
        }

        sortSpinner = Spinner(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                1f
            )
            adapter = ArrayAdapter(
                this@MainActivity,
                android.R.layout.simple_spinner_item,
                arrayOf("algo", "recent", "top")
            )
        }
        topBar.addView(sortSpinner)

        val searchButton = Button(this).apply {
            text = "Search"
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
            setOnClickListener {
                startActivity(Intent(this@MainActivity, SearchActivity::class.java))
            }
        }
        topBar.addView(searchButton)

        container.addView(topBar)

        progressBar = ProgressBar(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
            visibility = View.VISIBLE
        }
        container.addView(progressBar)

        errorText = TextView(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
            visibility = View.GONE
            textSize = 16f
            setTextColor(0xFFFF0000.toInt())
        }
        container.addView(errorText)

        recyclerView = RecyclerView(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT,
                1f
            )
            layoutManager = LinearLayoutManager(this@MainActivity)
            visibility = View.GONE
        }
        container.addView(recyclerView)

        loadMoreButton = Button(this).apply {
            text = "Load More"
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                80
            ).apply { setMargins(10, 10, 10, 10) }
            visibility = View.GONE
            setOnClickListener { loadMore() }
        }
        container.addView(loadMoreButton)

        return container
    }

    private fun setupUI() {
        adapter = RantAdapter(emptyList()) { rant ->
            val intent = Intent(this, DetailActivity::class.java)
            intent.putExtra("rantId", rant.id)
            startActivity(intent)
        }
        recyclerView.adapter = adapter

        sortSpinner.onItemSelectedListener = object : AdapterView.OnItemSelectedListener {
            override fun onItemSelected(parent: AdapterView<*>, view: View?, position: Int, id: Long) {
                val sort = parent.getItemAtPosition(position).toString()
                loadFeed(sort)
            }

            override fun onNothingSelected(parent: AdapterView<*>) {}
        }
    }

    private fun loadFeed(sort: String = "algo") {
        progressBar.visibility = View.VISIBLE
        recyclerView.visibility = View.GONE
        errorText.visibility = View.GONE
        loadMoreButton.visibility = View.GONE

        feedViewModel.loadFeed(sort, true)

        Thread.sleep(100)
        updateUI()
    }

    private fun loadMore() {
        if (isLoadingMore) return
        isLoadingMore = true
        loadMoreButton.isEnabled = false

        feedViewModel.loadMore()
        Thread.sleep(100)
        updateUI()
        isLoadingMore = false
        loadMoreButton.isEnabled = true
    }

    private fun updateUI() {
        when (feedViewModel.state) {
            is UiState.Loading -> {
                progressBar.visibility = View.VISIBLE
                recyclerView.visibility = View.GONE
                errorText.visibility = View.GONE
                loadMoreButton.visibility = View.GONE
            }
            is UiState.Success<*> -> {
                progressBar.visibility = View.GONE
                recyclerView.visibility = View.VISIBLE
                errorText.visibility = View.GONE
                loadMoreButton.visibility = View.VISIBLE
                adapter.updateRants(feedViewModel.rants)
            }
            is UiState.Error -> {
                progressBar.visibility = View.GONE
                recyclerView.visibility = View.GONE
                errorText.visibility = View.VISIBLE
                loadMoreButton.visibility = View.GONE
                val error = feedViewModel.state as UiState.Error
                errorText.text = "Error: ${error.exception.message}"
            }
        }
    }
}

class DetailActivity : Activity() {
    private lateinit var api: DevRantApi
    private lateinit var repository: RantRepository
    private lateinit var viewModel: DetailViewModel
    private lateinit var textView: TextView
    private lateinit var progressBar: ProgressBar
    private lateinit var errorText: TextView
    private lateinit var commentsContainer: LinearLayout
    private lateinit var scrollView: ScrollView

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val rantId = intent.getIntExtra("rantId", 0)

        api = DevRantApi()
        repository = RantRepository(api)
        viewModel = DetailViewModel(repository)

        setContentView(createDetailLayout())

        loadRant(rantId)
    }

    private fun createDetailLayout(): View {
        val container = LinearLayout(this).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = ViewGroup.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
        }

        val backButton = Button(this).apply {
            text = "Back"
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                80
            ).apply { setMargins(10, 10, 10, 10) }
            setOnClickListener { finish() }
        }
        container.addView(backButton)

        progressBar = ProgressBar(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
            visibility = View.VISIBLE
        }
        container.addView(progressBar)

        errorText = TextView(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
            visibility = View.GONE
            textSize = 16f
            setTextColor(0xFFFF0000.toInt())
        }
        container.addView(errorText)

        scrollView = ScrollView(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT,
                1f
            )
            visibility = View.GONE
        }

        val scrollContent = LinearLayout(this).apply {
            orientation = LinearLayout.VERTICAL
        }

        textView = TextView(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            ).apply { setMargins(15, 15, 15, 15) }
            textSize = 16f
        }
        scrollContent.addView(textView)

        commentsContainer = LinearLayout(this).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
        }
        scrollContent.addView(commentsContainer)

        scrollView.addView(scrollContent)
        container.addView(scrollView)

        return container
    }

    private fun loadRant(id: Int) {
        viewModel.loadRant(id)
        Thread.sleep(100)
        updateUI()
    }

    private fun updateUI() {
        when (viewModel.state) {
            is UiState.Loading -> {
                progressBar.visibility = View.VISIBLE
                scrollView.visibility = View.GONE
                errorText.visibility = View.GONE
            }
            is UiState.Success<*> -> {
                progressBar.visibility = View.GONE
                scrollView.visibility = View.VISIBLE
                errorText.visibility = View.GONE

                viewModel.rant?.let { rantFull ->
                    val rant = rantFull.rant
                    textView.text = "${rant.userUsername}\n\n${rant.text}\n\n++ ${rant.numUpvotes}  -- ${rant.numDownvotes}"

                    commentsContainer.removeAllViews()
                    rantFull.comments.forEach { comment ->
                        val commentView = TextView(this).apply {
                            text = "${comment.userUsername}: ${comment.body}\n(++ ${comment.numUpvotes}  -- ${comment.numDownvotes})"
                            textSize = 14f
                            layoutParams = LinearLayout.LayoutParams(
                                ViewGroup.LayoutParams.MATCH_PARENT,
                                ViewGroup.LayoutParams.WRAP_CONTENT
                            ).apply { setMargins(15, 10, 15, 10) }
                            setBackgroundColor(0xFFF5F5F5.toInt())
                            setPadding(10, 10, 10, 10)
                        }
                        commentsContainer.addView(commentView)
                    }
                }
            }
            is UiState.Error -> {
                progressBar.visibility = View.GONE
                scrollView.visibility = View.GONE
                errorText.visibility = View.VISIBLE
                val error = viewModel.state as UiState.Error
                errorText.text = "Error: ${error.exception.message}"
            }
        }
    }
}

class SearchActivity : Activity() {
    private lateinit var api: DevRantApi
    private lateinit var repository: RantRepository
    private lateinit var viewModel: SearchViewModel
    private lateinit var searchInput: EditText
    private lateinit var recyclerView: RecyclerView
    private lateinit var adapter: RantAdapter
    private lateinit var progressBar: ProgressBar

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        api = DevRantApi()
        repository = RantRepository(api)
        viewModel = SearchViewModel(repository)

        setContentView(createSearchLayout())
    }

    private fun createSearchLayout(): View {
        val container = LinearLayout(this).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = ViewGroup.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
        }

        val backButton = Button(this).apply {
            text = "Back"
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                80
            ).apply { setMargins(10, 10, 10, 10) }
            setOnClickListener { finish() }
        }
        container.addView(backButton)

        val searchBar = LinearLayout(this).apply {
            orientation = LinearLayout.HORIZONTAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                100
            ).apply { setMargins(10, 10, 10, 10) }
        }

        searchInput = EditText(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                1f
            )
            hint = "Search rants..."
        }
        searchBar.addView(searchInput)

        val searchButton = Button(this).apply {
            text = "Search"
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
            setOnClickListener {
                val term = searchInput.text.toString()
                if (term.isNotEmpty()) {
                    performSearch(term)
                }
            }
        }
        searchBar.addView(searchButton)
        container.addView(searchBar)

        progressBar = ProgressBar(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                200
            )
            visibility = View.GONE
        }
        container.addView(progressBar)

        adapter = RantAdapter(emptyList()) { rant ->
            val intent = Intent(this, DetailActivity::class.java)
            intent.putExtra("rantId", rant.id)
            startActivity(intent)
        }

        recyclerView = RecyclerView(this).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT,
                1f
            )
            layoutManager = LinearLayoutManager(this@SearchActivity)
            adapter = this@SearchActivity.adapter
        }
        container.addView(recyclerView)

        return container
    }

    private fun performSearch(term: String) {
        progressBar.visibility = View.VISIBLE
        viewModel.search(term)
        Thread.sleep(100)
        updateUI()
    }

    private fun updateUI() {
        when (viewModel.state) {
            is UiState.Loading -> {
                progressBar.visibility = View.VISIBLE
            }
            is UiState.Success<*> -> {
                progressBar.visibility = View.GONE
                adapter.updateRants(viewModel.results)
            }
            is UiState.Error -> {
                progressBar.visibility = View.GONE
            }
        }
    }
}

class RantAdapter(
    private var rants: List<Rant>,
    private val onItemClick: (Rant) -> Unit
) : RecyclerView.Adapter<RantViewHolder>() {

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RantViewHolder {
        val container = LinearLayout(parent.context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = ViewGroup.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
            setBackgroundColor(0xFFFAFAFA.toInt())
            setPadding(15, 15, 15, 15)
        }

        val titleText = TextView(parent.context).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
            textSize = 18f
            setTextColor(0xFF333333.toInt())
            tag = "title"
        }
        container.addView(titleText)

        val scoreText = TextView(parent.context).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            ).apply { setMargins(0, 10, 0, 0) }
            textSize = 14f
            setTextColor(0xFF666666.toInt())
            tag = "score"
        }
        container.addView(scoreText)

        val divider = View(parent.context).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                2
            ).apply { setMargins(0, 15, 0, 0) }
            setBackgroundColor(0xFFDDDDDD.toInt())
        }
        container.addView(divider)

        container.setOnClickListener { rant ->
            onItemClick(rant.tag as Rant)
        }

        return RantViewHolder(container)
    }

    override fun onBindViewHolder(holder: RantViewHolder, position: Int) {
        val rant = rants[position]
        holder.bind(rant)
    }

    override fun getItemCount(): Int = rants.size

    fun updateRants(newRants: List<Rant>) {
        rants = newRants
        notifyDataSetChanged()
    }
}

class RantViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView) {
    fun bind(rant: Rant) {
        val container = itemView as LinearLayout
        val titleText = container.findViewWithTag<TextView>("title")
        val scoreText = container.findViewWithTag<TextView>("score")

        titleText.text = "${rant.userUsername}: ${rant.text.take(100)}..."
        scoreText.text = "Score: ${rant.score} | Comments: ${rant.numComments}"
        container.tag = rant
    }
}
