package com.devrant.android

import android.content.Context
import android.net.Uri
import java.io.*
import java.net.HttpURLConnection
import java.net.URL
import java.util.concurrent.ConcurrentHashMap

data class Rant(
    val id: Int,
    val text: String,
    val numUpvotes: Int,
    val numDownvotes: Int,
    val score: Int,
    val createdTime: Long,
    val attachedImage: RantImage?,
    val numComments: Int,
    val tags: List<String>,
    val voteState: Int,
    val userId: Int,
    val userUsername: String,
    val userScore: Int
)

data class RantImage(
    val url: String,
    val width: Int,
    val height: Int
)

data class RantFull(
    val rant: Rant,
    val comments: List<Comment>,
    val success: Boolean
)

data class Comment(
    val id: Int,
    val rantId: Int,
    val body: String,
    val numUpvotes: Int,
    val numDownvotes: Int,
    val score: Int,
    val createdTime: Long,
    val voteState: Int,
    val userId: Int,
    val userUsername: String,
    val userScore: Int
)

data class UserProfile(
    val username: String,
    val score: Int,
    val about: String,
    val location: String,
    val createdTime: Long,
    val skills: String,
    val github: String,
    val avatar: String,
    val content: UserContent
)

data class UserContent(
    val rants: List<Rant>,
    val upvoted: List<Rant>,
    val comments: List<Comment>,
    val counts: ContentCounts
)

data class ContentCounts(
    val rants: Int,
    val upvoted: Int,
    val comments: Int
)

data class FeedResponse(
    val rants: List<Rant>,
    val success: Boolean
)

data class SearchResponse(
    val results: List<Rant>,
    val success: Boolean
)

class DevRantApi {
    private val baseUrl = "https://devrant.com/api/devrant"
    private val cache = ConcurrentHashMap<String, String>()
    private val cacheTimeouts = ConcurrentHashMap<String, Long>()
    private val cacheTimeout = 5 * 60 * 1000

    fun getRants(
        sort: String = "algo",
        limit: Int = 50,
        skip: Int = 0,
        callback: (List<Rant>?, Exception?) -> Unit
    ) {
        val url = "$baseUrl/rants?sort=$sort&limit=$limit&skip=$skip"
        makeRequest(url) { response, error ->
            if (error != null) {
                callback(null, error)
                return@makeRequest
            }
            try {
                val rants = parseRantsList(response)
                callback(rants, null)
            } catch (e: Exception) {
                callback(null, e)
            }
        }
    }

    fun getRant(
        id: Int,
        callback: (RantFull?, Exception?) -> Unit
    ) {
        val url = "$baseUrl/rants/$id"
        makeRequest(url) { response, error ->
            if (error != null) {
                callback(null, error)
                return@makeRequest
            }
            try {
                val rant = parseRantFull(response)
                callback(rant, null)
            } catch (e: Exception) {
                callback(null, e)
            }
        }
    }

    fun searchRants(
        term: String,
        callback: (List<Rant>?, Exception?) -> Unit
    ) {
        val url = "$baseUrl/search?term=${Uri.encode(term)}"
        makeRequest(url) { response, error ->
            if (error != null) {
                callback(null, error)
                return@makeRequest
            }
            try {
                val rants = parseSearchResults(response)
                callback(rants, null)
            } catch (e: Exception) {
                callback(null, e)
            }
        }
    }

    fun getUserProfile(
        username: String,
        callback: (UserProfile?, Exception?) -> Unit
    ) {
        val url = "$baseUrl/users/$username"
        makeRequest(url) { response, error ->
            if (error != null) {
                callback(null, error)
                return@makeRequest
            }
            try {
                val profile = parseUserProfile(response)
                callback(profile, null)
            } catch (e: Exception) {
                callback(null, e)
            }
        }
    }

    fun voteRant(
        id: Int,
        vote: Int,
        callback: (Boolean, Exception?) -> Unit
    ) {
        val url = "$baseUrl/rants/$id/vote?vote=$vote"
        makePostRequest(url, "") { response, error ->
            if (error != null) {
                callback(false, error)
                return@makePostRequest
            }
            try {
                callback(response?.contains("\"success\":true") ?: false, null)
            } catch (e: Exception) {
                callback(false, e)
            }
        }
    }

    fun voteComment(
        id: Int,
        vote: Int,
        callback: (Boolean, Exception?) -> Unit
    ) {
        val url = "$baseUrl/comments/$id/vote?vote=$vote"
        makePostRequest(url, "") { response, error ->
            if (error != null) {
                callback(false, error)
                return@makePostRequest
            }
            try {
                callback(response?.contains("\"success\":true") ?: false, null)
            } catch (e: Exception) {
                callback(false, e)
            }
        }
    }

    private fun makeRequest(
        url: String,
        callback: (String?, Exception?) -> Unit
    ) {
        val cacheKey = url
        val cached = cache[cacheKey]
        val cacheTime = cacheTimeouts[cacheKey] ?: 0

        if (cached != null && System.currentTimeMillis() - cacheTime < cacheTimeout) {
            callback(cached, null)
            return
        }

        Thread {
            try {
                val connection = URL(url).openConnection() as HttpURLConnection
                connection.requestMethod = "GET"
                connection.connectTimeout = 10000
                connection.readTimeout = 10000
                connection.setRequestProperty("User-Agent", "DevRant-Android-Client/1.0")

                val responseCode = connection.responseCode
                if (responseCode == HttpURLConnection.HTTP_OK) {
                    val response = connection.inputStream.bufferedReader().use { it.readText() }
                    cache[cacheKey] = response
                    cacheTimeouts[cacheKey] = System.currentTimeMillis()
                    callback(response, null)
                } else {
                    callback(null, Exception("HTTP $responseCode"))
                }
                connection.disconnect()
            } catch (e: Exception) {
                callback(null, e)
            }
        }.start()
    }

    private fun makePostRequest(
        url: String,
        body: String,
        callback: (String?, Exception?) -> Unit
    ) {
        Thread {
            try {
                val connection = URL(url).openConnection() as HttpURLConnection
                connection.requestMethod = "POST"
                connection.connectTimeout = 10000
                connection.readTimeout = 10000
                connection.setRequestProperty("Content-Type", "application/json")
                connection.setRequestProperty("User-Agent", "DevRant-Android-Client/1.0")
                connection.doOutput = true

                if (body.isNotEmpty()) {
                    connection.outputStream.write(body.toByteArray())
                }

                val responseCode = connection.responseCode
                if (responseCode in 200..299) {
                    val response = connection.inputStream.bufferedReader().use { it.readText() }
                    callback(response, null)
                } else {
                    callback(null, Exception("HTTP $responseCode"))
                }
                connection.disconnect()
            } catch (e: Exception) {
                callback(null, e)
            }
        }.start()
    }

    private fun parseRantsList(json: String): List<Rant> {
        val rants = mutableListOf<Rant>()
        val rantPattern = "\"id\":(\\d+).*?\"text\":\"([^\"]*?)\".*?\"num_upvotes\":(\\d+).*?\"num_downvotes\":(\\d+).*?\"score\":(\\d+).*?\"created_time\":(\\d+).*?\"num_comments\":(\\d+).*?\"user_id\":(\\d+).*?\"user_username\":\"([^\"]+)\".*?\"user_score\":(\\d+)".toRegex()

        for (match in rantPattern.findAll(json)) {
            val (id, text, upvotes, downvotes, score, createdTime, numComments, userId, username, userScore) = match.destructured
            rants.add(
                Rant(
                    id = id.toInt(),
                    text = unescapeJson(text),
                    numUpvotes = upvotes.toInt(),
                    numDownvotes = downvotes.toInt(),
                    score = score.toInt(),
                    createdTime = createdTime.toLong(),
                    attachedImage = null,
                    numComments = numComments.toInt(),
                    tags = emptyList(),
                    voteState = 0,
                    userId = userId.toInt(),
                    userUsername = username,
                    userScore = userScore.toInt()
                )
            )
        }
        return rants
    }

    private fun parseRantFull(json: String): RantFull {
        val rantMatch = "\"rant\":\\{(.*?)\\},\"comments\"".toRegex().find(json)
        val commentsMatch = "\"comments\":\\[(.*?)\\],\"success\"".toRegex().find(json)

        val rant = if (rantMatch != null) {
            parseRantFromJson(rantMatch.groupValues[1])
        } else {
            Rant(0, "", 0, 0, 0, 0, null, 0, emptyList(), 0, 0, "", 0)
        }

        val comments = mutableListOf<Comment>()
        if (commentsMatch != null) {
            val commentPattern = "\"id\":(\\d+).*?\"body\":\"([^\"]*?)\".*?\"num_upvotes\":(\\d+).*?\"num_downvotes\":(\\d+).*?\"user_username\":\"([^\"]+)\"".toRegex()
            for (match in commentPattern.findAll(commentsMatch.groupValues[1])) {
                val (id, body, upvotes, downvotes, username) = match.destructured
                comments.add(
                    Comment(
                        id = id.toInt(),
                        rantId = rant.id,
                        body = unescapeJson(body),
                        numUpvotes = upvotes.toInt(),
                        numDownvotes = downvotes.toInt(),
                        score = upvotes.toInt() - downvotes.toInt(),
                        createdTime = 0,
                        voteState = 0,
                        userId = 0,
                        userUsername = username,
                        userScore = 0
                    )
                )
            }
        }

        return RantFull(rant, comments, true)
    }

    private fun parseRantFromJson(json: String): Rant {
        val idPattern = "\"id\":(\\d+)".toRegex()
        val textPattern = "\"text\":\"([^\"]*?)\"".toRegex()
        val upvotesPattern = "\"num_upvotes\":(\\d+)".toRegex()
        val downvotesPattern = "\"num_downvotes\":(\\d+)".toRegex()
        val scorePattern = "\"score\":(\\d+)".toRegex()
        val createdPattern = "\"created_time\":(\\d+)".toRegex()
        val commentsPattern = "\"num_comments\":(\\d+)".toRegex()
        val userIdPattern = "\"user_id\":(\\d+)".toRegex()
        val userPattern = "\"user_username\":\"([^\"]+)\"".toRegex()
        val scoreUserPattern = "\"user_score\":(\\d+)".toRegex()

        return Rant(
            id = idPattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            text = unescapeJson(textPattern.find(json)?.groupValues?.get(1) ?: ""),
            numUpvotes = upvotesPattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            numDownvotes = downvotesPattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            score = scorePattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            createdTime = createdPattern.find(json)?.groupValues?.get(1)?.toLong() ?: 0,
            attachedImage = null,
            numComments = commentsPattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            tags = emptyList(),
            voteState = 0,
            userId = userIdPattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            userUsername = userPattern.find(json)?.groupValues?.get(1) ?: "",
            userScore = scoreUserPattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0
        )
    }

    private fun parseSearchResults(json: String): List<Rant> {
        return parseRantsList(json)
    }

    private fun parseUserProfile(json: String): UserProfile {
        val usernamePattern = "\"username\":\"([^\"]+)\"".toRegex()
        val scorePattern = "\"score\":(\\d+)".toRegex()
        val aboutPattern = "\"about\":\"([^\"]*?)\"".toRegex()
        val locationPattern = "\"location\":\"([^\"]*?)\"".toRegex()

        return UserProfile(
            username = usernamePattern.find(json)?.groupValues?.get(1) ?: "",
            score = scorePattern.find(json)?.groupValues?.get(1)?.toInt() ?: 0,
            about = unescapeJson(aboutPattern.find(json)?.groupValues?.get(1) ?: ""),
            location = unescapeJson(locationPattern.find(json)?.groupValues?.get(1) ?: ""),
            createdTime = 0,
            skills = "",
            github = "",
            avatar = "",
            content = UserContent(emptyList(), emptyList(), emptyList(), ContentCounts(0, 0, 0))
        )
    }

    private fun unescapeJson(s: String): String {
        return s
            .replace("\\\"", "\"")
            .replace("\\\\", "\\")
            .replace("\\n", "\n")
            .replace("\\r", "\r")
            .replace("\\t", "\t")
    }
}

class RantRepository(private val api: DevRantApi) {
    fun getFeed(sort: String = "algo", limit: Int = 50, skip: Int = 0, callback: (List<Rant>?, Exception?) -> Unit) {
        api.getRants(sort, limit, skip, callback)
    }

    fun getRantDetail(id: Int, callback: (RantFull?, Exception?) -> Unit) {
        api.getRant(id, callback)
    }

    fun search(term: String, callback: (List<Rant>?, Exception?) -> Unit) {
        api.searchRants(term, callback)
    }

    fun getUserProfile(username: String, callback: (UserProfile?, Exception?) -> Unit) {
        api.getUserProfile(username, callback)
    }

    fun voteRant(id: Int, vote: Int, callback: (Boolean, Exception?) -> Unit) {
        api.voteRant(id, vote, callback)
    }

    fun voteComment(id: Int, vote: Int, callback: (Boolean, Exception?) -> Unit) {
        api.voteComment(id, vote, callback)
    }
}

sealed class UiState {
    object Loading : UiState()
    data class Success<T>(val data: T) : UiState()
    data class Error(val exception: Exception) : UiState()
}

class FeedViewModel(private val repository: RantRepository) {
    private var currentSort = "algo"
    private var currentSkip = 0
    private val rants = mutableListOf<Rant>()

    var state: UiState = UiState.Loading
    var rants: List<Rant> = emptyList()

    fun loadFeed(sort: String = "algo", reset: Boolean = false) {
        if (reset) {
            rants.clear()
            currentSkip = 0
            currentSort = sort
        }

        state = UiState.Loading
        repository.getFeed(currentSort, 50, currentSkip) { items, error ->
            if (error != null) {
                state = UiState.Error(error)
            } else if (items != null) {
                rants.addAll(items)
                this.rants = rants.toList()
                state = UiState.Success(this.rants)
                currentSkip += items.size
            }
        }
    }

    fun loadMore() {
        loadFeed(currentSort, false)
    }
}

class DetailViewModel(private val repository: RantRepository) {
    var state: UiState = UiState.Loading
    var rant: RantFull? = null

    fun loadRant(id: Int) {
        state = UiState.Loading
        repository.getRantDetail(id) { detail, error ->
            if (error != null) {
                state = UiState.Error(error)
            } else if (detail != null) {
                rant = detail
                state = UiState.Success(detail)
            }
        }
    }

    fun voteRant(id: Int, vote: Int) {
        repository.voteRant(id, vote) { success, error ->
            if (success) {
                rant?.rant?.let {
                    val updated = it.copy(voteState = vote)
                    rant = rant?.copy(rant = updated)
                }
            }
        }
    }

    fun voteComment(id: Int, vote: Int) {
        repository.voteComment(id, vote) { success, error ->
            if (success) {
                state = UiState.Success(rant)
            }
        }
    }
}

class SearchViewModel(private val repository: RantRepository) {
    var state: UiState = UiState.Loading
    var results: List<Rant> = emptyList()

    fun search(term: String) {
        if (term.isEmpty()) {
            results = emptyList()
            state = UiState.Success(results)
            return
        }

        state = UiState.Loading
        repository.search(term) { items, error ->
            if (error != null) {
                state = UiState.Error(error)
            } else if (items != null) {
                results = items
                state = UiState.Success(items)
            }
        }
    }
}

class ProfileViewModel(private val repository: RantRepository) {
    var state: UiState = UiState.Loading
    var profile: UserProfile? = null

    fun loadProfile(username: String) {
        state = UiState.Loading
        repository.getUserProfile(username) { userProfile, error ->
            if (error != null) {
                state = UiState.Error(error)
            } else if (userProfile != null) {
                profile = userProfile
                state = UiState.Success(userProfile)
            }
        }
    }
}
