# Ultimate devRant Android Client - Setup Guide

## Project Structure

```
devrant-android/
├── app/
│   ├── src/
│   │   ├── main/
│   │   │   ├── java/com/devrant/android/
│   │   │   │   ├── DevRantApi.kt          # API client & data models
│   │   │   │   └── ui/
│   │   │   │       └── Activities.kt       # UI Activities & Adapters
│   │   │   └── AndroidManifest.xml
│   │   └── res/
│   │       ├── values/
│   │       │   ├── strings.xml
│   │       │   └── styles.xml
│   │       └── drawable/
│   │           └── ic_launcher.xml
│   └── build.gradle.kts
├── build.gradle.kts
└── settings.gradle.kts
```

## Setup Steps

### 1. Create Project
```bash
mkdir devrant-android && cd devrant-android
```

### 2. Create build.gradle.kts (Project Root)
```kotlin
plugins {
    id("com.android.application") version "8.1.0" apply false
}

buildscript {
    repositories {
        google()
        mavenCentral()
    }
}
```

### 3. Create settings.gradle.kts
```kotlin
pluginManagement {
    repositories {
        gradlePluginPortal()
        google()
        mavenCentral()
    }
}
dependencyResolutionManagement {
    repositoriesMode.set(RepositoriesMode.FAIL_ON_PROJECT_REPOS)
    repositories {
        google()
        mavenCentral()
    }
}
rootProject.name = "devrant-android"
include(":app")
```

### 4. Create app/build.gradle.kts
```kotlin
plugins {
    id("com.android.application")
    kotlin("android")
}

android {
    namespace = "com.devrant.android"
    compileSdk = 34

    defaultConfig {
        applicationId = "com.devrant.android"
        minSdk = 26
        targetSdk = 34
        versionCode = 1
        versionName = "1.0.0"
    }

    buildTypes {
        release {
            isMinifyEnabled = true
            proguardFiles(getDefaultProguardFile("proguard-android-optimize.txt"))
        }
    }

    compileOptions {
        sourceCompatibility = JavaVersion.VERSION_11
        targetCompatibility = JavaVersion.VERSION_11
    }

    kotlinOptions {
        jvmTarget = "11"
    }
}

dependencies {
    implementation("androidx.appcompat:appcompat:1.6.1")
    implementation("androidx.recyclerview:recyclerview:1.3.1")
    implementation("com.google.android.material:material:1.9.0")
    implementation("org.jetbrains.kotlin:kotlin-stdlib:1.9.0")
}
```

### 5. Create String Resources
File: `app/src/main/res/values/strings.xml`
```xml
<?xml version="1.0" encoding="utf-8"?>
<resources>
    <string name="app_name">devRant</string>
    <string name="sort_algo">Algorithm</string>
    <string name="sort_recent">Recent</string>
    <string name="sort_top">Top</string>
    <string name="search">Search</string>
    <string name="load_more">Load More</string>
</resources>
```

### 6. Create Theme
File: `app/src/main/res/values/styles.xml`
```xml
<?xml version="1.0" encoding="utf-8"?>
<resources>
    <style name="AppTheme" parent="Theme.AppCompat.Light.DarkActionBar">
        <item name="colorPrimary">@color/primary</item>
        <item name="colorPrimaryDark">@color/primary_dark</item>
        <item name="colorAccent">@color/accent</item>
    </style>
</resources>
```

File: `app/src/main/res/values/colors.xml`
```xml
<?xml version="1.0" encoding="utf-8"?>
<resources>
    <color name="primary">#FF6B00</color>
    <color name="primary_dark">#E55100</color>
    <color name="accent">#FFC107</color>
</resources>
```

## Build & Run

### Using Android Studio
1. Open Android Studio
2. File → Open → Select project root
3. Wait for Gradle sync
4. Run → Run 'app'

### Using Command Line
```bash
# Build
./gradlew build

# Install on device/emulator
./gradlew installDebug

# Run
adb shell am start -n com.devrant.android/.ui.MainActivity
```

## Features Implemented

### Feed Screen
- Display rants with algorithm, recent, or top sorting
- Infinite scroll with load-more button
- Tap rant to view details
- Shows score and comment count

### Detail Screen
- Full rant text with user info
- All comments with scores
- Upvote/downvote support (API ready)
- Back navigation

### Search Screen
- Search rants by keyword
- Real-time results
- Tap to view rant details

### Architecture
- **DevRantApi**: HTTP client with built-in caching
- **RantRepository**: Data access abstraction
- **ViewModels**: State management for each screen
- **Activities**: UI layer with manual layout creation
- **Adapters**: RecyclerView adapters for lists

## API Endpoints

### Main Endpoints Used
```
GET  /api/devrant/rants                    # Feed
GET  /api/devrant/rants/{id}              # Single rant with comments
GET  /api/devrant/search?term={query}     # Search
GET  /api/devrant/users/{username}        # User profile
POST /api/devrant/rants/{id}/vote         # Vote on rant
POST /api/devrant/comments/{id}/vote      # Vote on comment
```

### Base URL
```
https://devrant.com/api/devrant
```

## Implementation Details

### Network
- Uses native `HttpURLConnection`
- No third-party networking libraries
- Built-in caching with 5-minute timeout
- Automatic thread management

### UI
- Kotlin with no layout XML files
- Programmatic view creation
- LinearLayout based layouts
- RecyclerView for efficient scrolling

### Parsing
- Regex-based JSON parsing (no Gson/Moshi)
- Manual data class mapping
- Handles API quirks and edge cases

### State Management
- MVVM-like pattern with ViewModels
- UiState sealed class for state representation
- Callback-based async operations

## Customization

### Change API Base URL
Edit in `DevRantApi.kt`:
```kotlin
private val baseUrl = "https://devrant.com/api/devrant"
```

### Add Theme Support
Create `values-night/colors.xml` for dark mode colors.

### Modify Caching
In `DevRantApi.kt`:
```kotlin
private val cacheTimeout = 5 * 60 * 1000  // Change timeout in ms
```

### Add More Features
1. Create new ViewModel in `DevRantApi.kt`
2. Add new Activity in `Activities.kt`
3. Update `AndroidManifest.xml`
4. Add API methods as needed

## Important Notes

1. **Attribution Required**: Display devRant username when showing rant content per API guidelines
2. **No Authentication**: Current API doesn't require authentication for reading
3. **Rate Limiting**: Be mindful of API rate limits during development
4. **HTTP Access**: App uses cleartext for devrant.com - adjust manifest if needed

## Troubleshooting

### Build Fails
- Ensure SDK 34+ is installed
- Run: `./gradlew clean build`

### Network Issues
- Check internet permission in manifest
- Verify firewall/proxy settings
- Test API endpoint manually: `curl https://devrant.com/api/devrant/rants`

### UI Layout Issues
- Increase padding values in `LinearLayout.LayoutParams`
- Adjust text sizes in `dp` units
- Test on multiple screen sizes

## Future Enhancements

- [ ] User authentication (when API supports it)
- [ ] Posting rants and comments
- [ ] Image caching and display
- [ ] Dark mode theme
- [ ] Offline mode with Room database
- [ ] Notifications for followed users
- [ ] Advanced search filters
- [ ] User profiles and following

## Attribution

This app uses the unofficial devRant API. devRant was created by dfox and aVirgulata.

Official devRant: https://devrant.com
